<?php
/*
* 2007-2010 PrestaShop 
*
* NOTICE OF LICENSE
*
* This source file is subject to the Open Software License (OSL 3.0)
* that is bundled with this package in the file LICENSE.txt.
* It is also available through the world-wide-web at this URL:
* http://opensource.org/licenses/osl-3.0.php
* If you did not receive a copy of the license and are unable to
* obtain it through the world-wide-web, please send an email
* to license@prestashop.com so we can send you a copy immediately.
*
* DISCLAIMER
*
* Do not edit or add to this file if you wish to upgrade PrestaShop to newer
* versions in the future. If you wish to customize PrestaShop for your
* needs please refer to http://www.prestashop.com for more information.
*
*  @author Prestashop SA <contact@prestashop.com>
*  @copyright  2007-2010 Prestashop SA
*  @license    http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
*  International Registered Trademark & Property of PrestaShop SA
*/

class WebserviceRequestCore
{
	protected $_available_languages = null;
	/** 
	 * Errors triggered at execution
	 * @var array
	 */
	public $errors = array();
	
	/**
	 * Set if return should display content or not
	 * @var boolean
	 */
	protected $_outputEnabled = true;
	
	/**
	 * Set if the management is specific or if it is classic (entity management)
	 * @var boolean
	 */
	protected $objectSpecificManagement = false;
	
	/**
	 * Base PrestaShop webservice URL
	 * @var string
	 */
	public $wsUrl;
	
	/**
	 * PrestaShop Webservice Documentation URL
	 * @var string 
	 */
	protected $_docUrl = 'http://prestashop.com/docs/1.4/webservice';
	
	/**
	 * Set if the authentication key was checked
	 * @var boolean
	 */
	protected $_authenticated = false;
	
	/**
	 * HTTP Method to support
	 * @var string
	 */
	public $method;
	
	/**
	 * The segment of the URL
	 * @var array
	 */
	public $urlSegment = array();
	
	/**
	 * The segment list of the URL after the "api" segment
	 * @var array
	 */
	public $urlFragments = array();
	
	/**
	 * The time in microseconds of the start of the execution of the web service request
	 * @var int
	 */
	protected $_startTime = 0;
	
	/**
	 * The list of each resources manageable via web service
	 * @var array
	 */
	public $resourceList;
	
	/**
	 * The configuration parameters of the current resource
	 * @var array
	 */
	public $resourceConfiguration;
	
	/**
	 * The permissions for the current key
	 * @var array
	 */
	public $keyPermissions;
	
	/**
	 * The XML string to display if web service call succeed
	 * @var string
	 */
	protected $specificOutput = '';
	
	/**
	 * The list of objects to display
	 * @var array
	 */
	public $objects;
	
	/**
	 * The current object to support, it extends the PrestaShop ObjectModel
	 * @var ObjectModel
	 */
	protected $_object;
	
	/**
	 * The schema to display. 
	 * If null, no schema have to be displayed and normal management has to be performed
	 * @var string 
	 */
	public $schemaToDisplay;
	
	/**
	 * The fields to display. These fields will be displayed when retrieving objects
	 * @var string
	 */
	public $fieldsToDisplay = 'minimum';
	
	/**
	 * If we are in PUT or POST case, we use this attribute to store the xml string value during process
	 * @var string
	 */
	protected $_inputXml;
	
	/**
	 * Object instance for singleton
	 * @var WebserviceRequest
	 */
	protected static $_instance;
	
	/**
	 * Key used for authentication
	 * @var string
	 */
	protected $_key;
	
	/**
	 * This is used to have a deeper tree diagram.
	 * @var int
	 */
	public $depth = 0;
	
	/**
	 * Name of the output format
	 * @var string
	 */
	protected $outputFormat = 'xml';
	
	/**
	 * The object to build the output.
	 * @var WebserviceOutputBuilder
	 */
	protected $objOutput;
	
	/**
	 * Save the class name for override used in getInstance()
	 * @var ws_current_classname
	 */
	public static $ws_current_classname;
	
	public function getOutputEnabled()
	{
		return $this->_outputEnabled;
	}
	
	public function setOutputEnabled($bool)
	{
		if (Validate::isBool($bool))
			$this->_outputEnabled = $bool; 
		return $this;
	}
	
	/**
	 * Get WebserviceRequest object instance (Singleton)
	 *
	 * @return object WebserviceRequest instance
	 */
	public static function getInstance()
	{
		if (!isset(self::$_instance))
			self::$_instance = new WebserviceRequest::$ws_current_classname();
		return self::$_instance;
	}
	
	protected function getOutputObject($type)
	{
		switch ($type)
		{
			case 'JSON' :
				$obj_render = new WebserviceOutputJSON();
				break;
			case 'XML' :
			default :
				$obj_render = new WebserviceOutputXML();
				break;
		}
		return $obj_render;
	}
	
	public static function getResources()
	{
		$resources = array(
			'addresses' => array('description' => 'The Customer, Manufacturer and Customer addresses','class' => 'Address'),
			'carriers' => array('description' => 'The Carriers','class' => 'Carrier'),
			'carts' => array('description' => 'Customer\'s carts', 'class' => 'Cart'),
			'categories' => array('description' => 'The product categories','class' => 'Category'),
			'combinations' => array('description' => 'The product combinations','class' => 'Combination'),
			'configurations' => array('description' => 'Shop configuration', 'class' => 'Configuration'),
			'countries' => array('description' => 'The countries','class' => 'Country'),
			'currencies' => array('description' => 'The currencies', 'class' => 'Currency'),
			'customers' => array('description' => 'The e-shop\'s customers','class' => 'Customer'),
			'deliveries' => array('description' => 'Product delivery', 'class' => 'Delivery'),
			'groups' => array('description' => 'The customer\'s groups','class' => 'Group'),
			'guests' => array('description' => 'The guests', 'class' => 'Guest'),
			'images' => array('description' => 'The images', 'specific_management' => true),
			'image_types' => array('description' => 'The image types', 'class' => 'ImageType'),
			'languages' => array('description' => 'Shop languages', 'class' => 'Language'),
			'manufacturers' => array('description' => 'The product manufacturers','class' => 'Manufacturer'),
			'order_details' => array('description' => 'Details of an order', 'class' => 'OrderDetail'),
			'order_discounts' => array('description' => 'Discounts of an order', 'class' => 'OrderDiscount'),
			'order_histories' => array('description' => 'The Order histories','class' => 'OrderHistory'),
			'orders' => array('description' => 'The Customers orders','class' => 'Order'),
			'order_states' => array('description' => 'The Order states','class' => 'OrderState'),
			'price_ranges' => array('description' => 'Price ranges', 'class' => 'RangePrice'),
			'product_features' => array('description' => 'The product features','class' => 'Feature'),
			'product_feature_values' => array('description' => 'The product feature values','class' => 'FeatureValue'),
			'product_options' => array('description' => 'The product options','class' => 'AttributeGroup'),
			'product_option_values' => array('description' => 'The product options value','class' => 'Attribute'),
			'products' => array('description' => 'The products','class' => 'Product'),
			'states' => array('description' => 'The available states of countries','class' => 'State'),
			'stores' => array('description' => 'The stores', 'class' => 'Store'),
			'suppliers' => array('description' => 'The product suppliers','class' => 'Supplier'),
			'tags' => array('description' => 'The Products tags','class' => 'Tag'),
			'translated_configurations' => array('description' => 'Shop configuration', 'class' => 'TranslatedConfiguration'),
			'weight_ranges' => array('description' => 'Weight ranges', 'class' => 'RangeWeight'),
			'zones' => array('description' => 'The Countries zones','class' => 'Zone'),
			'employees' => array('description' => 'The Employees', 'class' => 'Employee'),
			'stock_movements' => array('description' => 'Stock movements management', 'class' => 'StockMvt', 'forbidden_method' => array('PUT')),
			'search' => array('description' => 'Search', 'specific_management' => true, 'forbidden_method' => array('PUT', 'POST', 'DELETE')),
			'stock_movement_reasons' => array('description' => 'The stock movement reason', 'class' => 'StockMvtReason'),
			'content_management_system' => array('description' => 'Content management system', 'class' => 'CMS'),
			'taxes' => array('description' => 'The tax rate', 'class' => 'Tax'),
			'tax_rule_groups' => array('description' => 'Tax rule groups', 'class' => 'TaxRulesGroup'),
			'customer_messages' => array('description' => 'Customer messages', 'class' => 'CustomerMessage'),
			'customer_threads' => array('description' => 'Customer threads', 'class' => 'CustomerThread'),
		);
		ksort($resources);
		return $resources;
	}
	
	// @todo Check how get parameters
	// @todo : set this method out
	/**
	 * This method is used for calculate the price for products on the output details
	 * 
	 * @param $field
	 * @param $entity_object
	 * @param $ws_params
	 * @return array field parameters.
	 */
	public function getPriceForProduct($field, $entity_object, $ws_params)
	{
		if (is_int($entity_object->id))
		{
			$arr_return = $this->specificPriceForProduct($entity_object, array('default_price' => ''));
			$field['value'] = $arr_return['default_price']['value'];
		}
		return $field;
	}
	
	// @todo : set this method out
	/**
	 * This method is used for calculate the price for products on a virtual fields
	 * 
	 * @param $entity_object
	 * @param array $parameters
	 * @return array
	 */
	public function specificPriceForProduct($entity_object, $parameters)
	{
		foreach (array_keys($parameters) as $name)
			$parameters[$name]['object_id'] = $entity_object->id;

		return $this->specificPriceCalculation($parameters);
	}

	public function specificPriceCalculation($parameters)
	{
		$arr_return = array();
		foreach ($parameters as $name => $value)
		{
			$id_country = (isset($value['country']) ? $value['country'] : (int)(_PS_COUNTRY_DEFAULT_));
			$id_state = (isset($value['state']) ? $value['state'] : 0);
			$id_currency = (isset($value['currency']) ? $value['currency'] : _PS_CURRENCY_DEFAULT_);
			$id_group = (isset($value['group']) ? $value['group'] : Configuration::get('_PS_DEFAULT_CUSTOMER_GROUP_'));
			$quantity = (isset($value['quantity']) ? $value['quantity'] : 1);
			$use_tax = (isset($value['use_tax']) ? $value['use_tax'] : Configuration::get('PS_TAX'));
			$decimals = (isset($value['decimals']) ? $value['decimals'] : Configuration::get('PS_PRICE_ROUND_MODE'));
			$id_product_attribute = (isset($value['product_attribute']) ? $value['product_attribute'] : null);
			$id_county = (isset($value['county']) ? $value['county'] : null);
			$only_reduc = (isset($value['only_reduction']) ? $value['only_reduction'] : false);
			$use_reduc = (isset($value['use_reduction']) ? $value['use_reduction'] : true);
			$use_ecotax = (isset($value['use_ecotax']) ? $value['use_ecotax'] : Configuration::get('PS_USE_ECOTAX'));
			$specific_price_output = null;
			$id_county = (isset($value['county']) ? $value['county'] : 0);
			$return_value = Product::priceCalculation(null, $value['object_id'], $id_product_attribute, $id_country, $id_state, 
			$id_county, $id_currency, $id_group, $quantity, $use_tax, $decimals, $only_reduc, $use_reduc, $use_ecotax, 
			$specific_price_output, null);
			$arr_return[$name] = array('sqlId' => strtolower($name), 'value' => sprintf('%f', $return_value));
		}
		return $arr_return;
	}
	// @todo : set this method out
	/**
	 * This method is used for calculate the price for products on a virtual fields
	 * 
	 * @param $entity_object
	 * @param array $parameters
	 * @return array
	 */
	public function specificPriceForCombination($entity_object, $parameters)
	{
		foreach (array_keys($parameters) as $name)
		{
			$parameters[$name]['object_id'] = $entity_object->id_product;
			$parameters[$name]['product_attribute'] = $entity_object->id;
		}

		return $this->specificPriceCalculation($parameters);
	}
	
	/**
	 * Start Webservice request
	 * 	Check webservice activation
	 * 	Check autentication
	 * 	Check resource
	 * 	Check HTTP Method
	 * 	Execute the action
	 * 	Display the result
	 *	
	 * @param string $key
	 * @param string $method
	 * @param string $url
	 * @param string $params
	 * @param string $inputXml
	 *
	 * @return array Returns an array of results (headers, content, type of resource...)
	 */
	public function fetch($key, $method, $url, $params, $bad_class_name, $inputXml = null)
	{
		// Time logger
		$this->_startTime = microtime(true);
		$this->objects = array();
		
		// Error handler
		set_error_handler(array($this, 'webserviceErrorHandler'));
		ini_set('html_errors', 'off');
		
		// Two global vars, for compatibility with the PS core...
		global $webservice_call, $display_errors;
		$webservice_call = true;
		$display_errors = strtolower(ini_get('display_errors')) != 'off';
		$this->wsUrl = Tools::getHttpHost(true).__PS_BASE_URI__.'api/';
		
		// set the output object which manage the content and header structure and informations
		$this->objOutput = new WebserviceOutputBuilder($this->wsUrl);
		$this->_key = trim($key);
		
		$this->outputFormat = isset($params['output_format']) ? $params['output_format'] : $this->outputFormat;
		// Set the render object to build the output on the asked format (XML, JSON, CSV, ...)
		$this->objOutput->setObjectRender($this->getOutputObject($this->outputFormat));
		
		// Check webservice activation and request authentication
		if ($this->webserviceChecks())
		{
			if ($bad_class_name)
				$this->setError(500, 'Bad override class name for this key. Please update class_name field', 126);
			// parse request url
			$this->method = $method;
			$this->urlSegment = explode('/', $url);
			$this->urlFragments = $params;
			$xml = file_get_contents('php://input');
			if (strncmp($xml, 'xml=', 4) == 0)
				$xml = substr($xml, 4);
			if (get_magic_quotes_gpc() == '1')
				$inputXml = stripslashes($inputXml);
			$this->_inputXml = $inputXml;
			$this->depth = isset($this->urlFragments['depth']) ? (int)$this->urlFragments['depth'] : $this->depth;
			
			
			try {
				// Method below set a particular fonction to use on the price field for products entity
				// @see WebserviceRequest::getPriceForProduct() method
				// @see WebserviceOutputBuilder::setSpecificField() method
				$this->objOutput->setSpecificField($this, 'getPriceForProduct', 'price', 'products');
				if (isset($this->urlFragments['price']))
				{
					$this->objOutput->setVirtualField($this, 'specificPriceForCombination', 'combinations', $this->urlFragments['price']);
					$this->objOutput->setVirtualField($this, 'specificPriceForProduct', 'products', $this->urlFragments['price']);
				}
			} catch (Exception $e) {
				$this->setError(500, $e->getMessage(), $e->getCode());
			}
			
			if (isset($this->urlFragments['language']))
				$this->_available_languages = $this->filterLanguage();
			else
			{
				foreach (Language::getLanguages() as $key=>$language)
					$this->_available_languages[] = $language['id_lang'];
			}
			
			if (empty($this->_available_languages))
				$this->setError(400, 'language is not available', 81);
			
			// Need to set available languages for the render object.
			// Thus we can filter i18n field for the output
			// @see WebserviceOutputXML::renderField() method for example
			$this->objOutput->objectRender->setLanguages($this->_available_languages);
			
			// check method and resource
			if (empty($this->errors) && $this->checkResource() && $this->checkHTTPMethod())
			{
				// The resource list is necessary for build the output
				$this->objOutput->setWsResources($this->resourceList);
				
				// if the resource is a core entity...
				if (!isset($this->resourceList[$this->urlSegment[0]]['specific_management']) || !$this->resourceList[$this->urlSegment[0]]['specific_management'])
				{
					
					// load resource configuration
					if ($this->urlSegment[0] != '')
					{
						$object = new $this->resourceList[$this->urlSegment[0]]['class']();
						if (isset($this->resourceList[$this->urlSegment[0]]['parameters_attribute']))
							$this->resourceConfiguration = $object->getWebserviceParameters($this->resourceList[$this->urlSegment[0]]['parameters_attribute']);
						else
							$this->resourceConfiguration = $object->getWebserviceParameters();
					}
					$success = false;
					// execute the action
					switch ($this->method)
					{
						case 'GET':
						case 'HEAD':
							if ($this->executeEntityGetAndHead())
								$success = true;
							break;
						case 'POST':
							if ($this->executeEntityPost())
								$success = true;
							break;
						case 'PUT':
							if ($this->executeEntityPut())
								$success = true;
							break;
						case 'DELETE':
							$this->executeEntityDelete();
							break;
					}
					// Need to set an object for the WebserviceOutputBuilder object in any case  
					// because schema need to get webserviceParameters of this object
					if (isset($object))
						$this->objects['empty'] = $object;
				}
				// if the management is specific
				else
				{
					$specificObjectName = 'WebserviceSpecificManagement'.ucfirst(Tools::toCamelCase($this->urlSegment[0]));
					if (!class_exists($specificObjectName))
						$this->setError(501, sprintf('The specific management class is not implemented for the "%s" entity.', $this->urlSegment[0]), 124);
					else
					{
						$this->setFieldsToDisplay();
						$this->objectSpecificManagement = new $specificObjectName();
						$this->objectSpecificManagement->setObjectOutput($this->objOutput)
													   ->setWsObject($this);
						
						try {
							$this->objectSpecificManagement->manage();
						} catch (WebserviceException $e) {
							if ($e->getType() == WebserviceException::DID_YOU_MEAN)
								$this->setErrorDidYouMean($e->getStatus(), $e->getMessage(), $e->getWrongValue(), $e->getAvailableValues(), $e->getCode());
							elseif ($e->getType() == WebserviceException::SIMPLE)
								$this->setError($e->getStatus(), $e->getMessage(), $e->getCode());
						}
					}
				}
			}
		}

		return $this->returnOutput();
	}
	
	protected function webserviceChecks()
	{
		return ($this->isActivated() && $this->authenticate());
	}
	
	/**
	 * Set a webservice error
	 *
	 * @param int $status
	 * @param string $label
	 * @param int $code
	 * @return void
	 */
	public function setError($status, $label, $code)
	{
		global $display_errors;
		$this->objOutput->setStatus($status);
		$this->errors[] = $display_errors ? array($code, $label) : 'Internal error. To see this error please display the PHP errors.';
	}
	
	/**
	 * Set a webservice error and propose a new value near from the available values
	 *
	 * @param int $num
	 * @param string $label
	 * @param array $value
	 * @param array $values
	 * @param int $code
	 * @return void
	 */
	public function setErrorDidYouMean($num, $label, $value, $available_values, $code)
	{
		$this->setError($num, $label.'. Did you mean: "'.$this->getClosest($value, $available_values).'"?'.(count($available_values) > 1 ? ' The full list is: "'.implode('", "', $available_values).'"' : ''), $code);
	}
	
	/**
	 * Return the nearest value picked in the values list
	 *
	 * @param string $input
	 * @param array $words
	 * @return string
	 */
	protected function getClosest($input, $words)
	{
		$shortest = -1;
		foreach ($words as $word)
		{
			$lev = levenshtein($input, $word);
			if ($lev == 0)
			{
				$closest = $word;
				$shortest = 0;
				break;
			}
			if ($lev <= $shortest || $shortest < 0)
			{
				$closest = $word;
				$shortest = $lev;
			}
		}
		return $closest;
	}
	
	/**
	 * Used to replace the default PHP error handler, in order to display PHP errors in a XML format 
	 *
	 * @param string $errno contains the level of the error raised, as an integer
	 * @param array $errstr contains the error message, as a string
	 * @param array $errfile errfile, which contains the filename that the error was raised in, as a string
	 * @param array $errline errline, which contains the line number the error was raised at, as an integer
	 * @return boolean Always return true to avoid the default PHP error handler
	 */
	public function webserviceErrorHandler($errno, $errstr, $errfile, $errline)
	{
		$display_errors = strtolower(ini_get('display_errors')) != 'off';
		if (!(error_reporting() & $errno) || $display_errors)
			return;
			
		$errortype = array (
                E_ERROR              => 'Error',
                E_WARNING            => 'Warning',
                E_PARSE              => 'Parse',
                E_NOTICE             => 'Notice',
                E_CORE_ERROR         => 'Core Error',
                E_CORE_WARNING       => 'Core Warning',
                E_COMPILE_ERROR      => 'Compile Error',
                E_COMPILE_WARNING    => 'Compile Warning',
                E_USER_ERROR         => 'Error',
                E_USER_WARNING       => 'User warning',
                E_USER_NOTICE        => 'User notice',
                E_STRICT             => 'Runtime Notice',
                E_RECOVERABLE_ERROR => 'Recoverable error'
                );
		$type = (isset($errortype[$errno]) ? $errortype[$errno] : 'Unknown error');
		error_log('[PHP '.$type.' #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')');
		switch($errno)
		{
			case E_ERROR:
				WebserviceRequest::getInstance()->setError(500, '[PHP Error #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 2);
				break;
			case E_WARNING:
				WebserviceRequest::getInstance()->setError(500, '[PHP Warning #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 3);
				break;
			case E_PARSE:
				WebserviceRequest::getInstance()->setError(500, '[PHP Parse #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 4);
				break;
			case E_NOTICE:
				WebserviceRequest::getInstance()->setError(500, '[PHP Notice #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 5);
				break;
			case E_CORE_ERROR:
				WebserviceRequest::getInstance()->setError(500, '[PHP Core #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 6);
				break;
			case E_CORE_WARNING:
				WebserviceRequest::getInstance()->setError(500, '[PHP Core warning #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 7);
				break;
			case E_COMPILE_ERROR:
				WebserviceRequest::getInstance()->setError(500, '[PHP Compile #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 8);
				break;
			case E_COMPILE_WARNING:
				WebserviceRequest::getInstance()->setError(500, '[PHP Compile warning #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 9);
				break;
			case E_USER_ERROR:
				WebserviceRequest::getInstance()->setError(500, '[PHP Error #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 10);
				break;
			case E_USER_WARNING:
				WebserviceRequest::getInstance()->setError(500, '[PHP User warning #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 11);
				break;
			case E_USER_NOTICE:
				WebserviceRequest::getInstance()->setError(500, '[PHP User notice #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 12);
				break;
			case E_STRICT:
				WebserviceRequest::getInstance()->setError(500, '[PHP Strict #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 13);
				break;
			case E_RECOVERABLE_ERROR:
				WebserviceRequest::getInstance()->setError(500, '[PHP Recoverable error #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 14);
				break;
			default:
				WebserviceRequest::getInstance()->setError(500, '[PHP Unknown error #'.$errno.'] '.$errstr.' ('.$errfile.', line '.$errline.')', 15);
		}
		return true;
	}
	
	/**
	 * Check if there is one or more error
	 *
	 * @return boolean
	 */
	protected function hasErrors()
	{
		return (boolean)$this->errors;
	}
	
	/**
	 * Check request authentication
	 *
	 * @return boolean
	 */
	protected function authenticate()
	{
		if (!$this->hasErrors())
		{
			if (is_null($this->_key))
			{
				$this->setError(401, 'Please enter the authentication key as the login. No password required', 16);
			}
			else
			{
				if (empty($this->_key))
				{
					$this->setError(401, 'Authentication key is empty', 17);
				}
				elseif (strlen($this->_key) != '32')
				{
					$this->setError(401, 'Invalid authentication key format', 18);
				}
				else
				{
					$keyValidation = WebserviceKey::isKeyActive($this->_key);
					if (is_null($keyValidation))
					{
						$this->setError(401, 'Authentification key does not exist', 19);
					}
					else if ($keyValidation === true)
					{
						$this->keyPermissions = WebserviceKey::getPermissionForAccount($this->_key);
					}
					else
					{
						$this->setError(401, 'Authentification key is not active', 20);
					}
					
					if (!$this->keyPermissions)
					{
						$this->setError(401, 'No permission for this authentication key', 21);
					}
				}
			}
			if ($this->hasErrors())
			{
				header('WWW-Authenticate: Basic realm="Welcome to PrestaShop Webservice, please enter the authentication key as the login. No password required."');
				$this->objOutput->setStatus(401);
				return false;
			}
			else
			{
				// only now we can say the access is authenticated
				$this->_authenticated = true;
				return true;
			}
		}
	}
	
	/**
	 * Check webservice activation
	 *
	 * @return boolean
	 */
	protected function isActivated()
	{
		if (!Configuration::get('PS_WEBSERVICE'))
		{
			$this->setError(503, 'The PrestaShop webservice is disabled. Please activate it in the PrestaShop Back Office', 22);
			return false;
		}
		return true;
	}
	
	/**
	 * Check HTTP method
	 *
	 * @return boolean
	 */
	protected function checkHTTPMethod()
	{
		if (!in_array($this->method, array('GET', 'POST', 'PUT', 'DELETE', 'HEAD')))
			$this->setError(405, 'Method '.$this->method.' is not valid', 23);
		elseif (isset($this->urlSegment[0]) && isset($this->resourceList[$this->urlSegment[0]]['forbidden_method']) && in_array($this->method, $this->resourceList[$this->urlSegment[0]]['forbidden_method']))
			$this->setError(405, 'Method '.$this->method.' is not allowed for the resource '.$this->urlSegment[0], 101);
		elseif ($this->urlSegment[0] && !in_array($this->method, $this->keyPermissions[$this->urlSegment[0]]))
			$this->setError(405, 'Method '.$this->method.' is not allowed for the resource '.$this->urlSegment[0].' with this authentication key', 25);
		else
			return true;
		return false;
	}
	
	/**
	 * Check resource validity
	 *
	 * @return boolean
	 */
	protected function checkResource()
	{
		$this->resourceList = $this->getResources();
		$resourceNames = array_keys($this->resourceList);
		if ($this->urlSegment[0] == '')
			$this->resourceConfiguration['objectsNodeName'] = 'resources';
		elseif (in_array($this->urlSegment[0], $resourceNames))
		{
			if (!in_array($this->urlSegment[0], array_keys($this->keyPermissions)))
			{
				$this->setError(401, 'Resource of type "'.$this->urlSegment[0].'" is not allowed with this authentication key', 26);
				return false;
			}
		}
		else
		{
			$this->setErrorDidYouMean(400, 'Resource of type "'.$this->urlSegment[0].'" does not exists', $this->urlSegment[0], $resourceNames, 27);
			return false;
		}
		return true;
	}
	
	
	protected function setObjects()
	{
		$objects = array();
		$arr_avoid_id = array();
		$ids = array();
		if (isset($this->urlFragments['id']))
		{
			preg_match('#^\[(.*)\]$#Ui', $this->urlFragments['id'], $matches);
			if (count($matches) > 1)
				$ids = explode(',', $matches[1]);
		}
		else
		{
			$ids[] = (int)$this->urlSegment[1];
		}
		if (!empty($ids))
		{
			foreach ($ids as $id)
			{
				$object = new $this->resourceConfiguration['retrieveData']['className']((int)$id);
				if (!$object->id)
					$arr_avoid_id[] = $id;
				else
					$objects[] = $object;
			}
		}
		
		if (!empty($arr_avoid_id) || empty($ids))
		{
			$this->setError(404, 'Id(s) not exists: '.implode(', ', $arr_avoid_id), 87);
			$this->_outputEnabled = true;
		}
		else 
		{
			
		}
	}
	
	protected function parseDisplayFields($str)
	{
		$bracket_level = 0;
		$part = array();
		$tmp = '';
		for ($i = 0; $i < strlen($str); $i++)
		{
			if ($str[$i] == ',' && $bracket_level == 0)
			{
				$part[] = $tmp;
				$tmp = '';
			}
			else
				$tmp .= $str[$i];
			if ($str[$i] == '[')
				$bracket_level++;
			if ($str[$i] == ']')
				$bracket_level--;
		}
		if ($tmp != '')
			$part[] = $tmp;
		$fields = array();
		foreach ($part as $str)
		{
			$field_name = trim(substr($str, 0, (strpos($str, '[') === false ? strlen($str) : strpos($str, '['))));
			if (!isset($fields[$field_name])) $fields[$field_name] = null;
			if (strpos($str, '[') !== false)
			{
				$sub_fields = substr($str, strpos($str, '[') + 1, strlen($str) - strpos($str, '[') - 2);
				$tmp_array = array();
				if (strpos($sub_fields, ',') !== false)
					$tmp_array = explode(',', $sub_fields);
				else
					$tmp_array = array($sub_fields);
				$fields[$field_name] = (is_array($fields[$field_name])) ? array_merge($fields[$field_name], $tmp_array) : $tmp_array;
			}
		}
		return $fields;
	}
	
	public function setFieldsToDisplay()
	{
		// set the fields to display in the list : "full", "minimum", "field_1", "field_1,field_2,field_3"
		if (isset($this->urlFragments['display']))
		{
			$this->fieldsToDisplay = $this->urlFragments['display'];
			if ($this->fieldsToDisplay != 'full' && $this->fieldsToDisplay != 'minimum')
			{
				preg_match('#^\[(.*)\]$#Ui', $this->fieldsToDisplay, $matches);
				if (count($matches))
				{
					$error = false;
					$fieldsToTest = $this->parseDisplayFields($matches[1]);
					foreach ($fieldsToTest as $field_name => $part)
					{
						// in case it is not an association
						if (!is_array($part))
						{
							// We have to allow new specific field for price calculation too
							$error = (!isset($this->resourceConfiguration['fields'][$field_name]) && !isset($this->urlFragments['price'][$field_name]));
						}
						else
						{
							// if this association does not exists
							if (!array_key_exists($field_name, $this->resourceConfiguration['associations'])) {
								$error = true;
							}
							foreach ($part as $field)
								if ($field != 'id' && !array_key_exists($field, $this->resourceConfiguration['associations'][$field_name]['fields']))
								{
									$error = true;
									break;
								}
						}
						if ($error)
						{
							$this->setError(400,'Unable to display this field "'.$field_name.(is_array($part) ? ' (details : '.var_export($part, true).')' : '').'". However, these are available: '.implode(', ', array_keys($this->resourceConfiguration['fields'])), 35);
							return false;
						}
					}
					$this->fieldsToDisplay = $fieldsToTest;
				}
				else
				{
					$this->setError(400, 'The \'display\' syntax is wrong. You can set \'full\' or \'[field_1,field_2,field_3,...]\'. These are available: '.implode(', ', array_keys($this->resourceConfiguration['fields'])), 36);
					return false;
				}
			}
		}
		return true;
	}
	
	protected function manageFilters()
	{
		// filtered fields which can not use filters : hidden_fields
		$available_filters = array();
		// filtered i18n fields which can use filters
		$i18n_available_filters = array();
		foreach ($this->resourceConfiguration['fields'] as $fieldName => $field)
			if ((!isset($this->resourceConfiguration['hidden_fields']) ||
				(isset($this->resourceConfiguration['hidden_fields']) && !in_array($fieldName, $this->resourceConfiguration['hidden_fields']))))
				if ((!isset($field['i18n']) || 
				(isset($field['i18n']) && !$field['i18n'])))
					$available_filters[] = $fieldName;
				else
					$i18n_available_filters[] = $fieldName;
		
		// Date feature : date=1
		if (!empty($this->urlFragments['date']) && $this->urlFragments['date'])
		{
			if (!in_array('date_add', $available_filters))
				$available_filters[] = 'date_add';
			if (!in_array('date_upd', $available_filters))
				$available_filters[] = 'date_upd';
			if (!array_key_exists('date_add', $this->resourceConfiguration['fields']))
				$this->resourceConfiguration['fields']['date_add'] = array('sqlId' => 'date_add');
			if (!array_key_exists('date_upd', $this->resourceConfiguration['fields']))
				$this->resourceConfiguration['fields']['date_upd'] = array('sqlId' => 'date_upd');
		}
		else
			foreach ($available_filters as $key => $value)
				if ($value == 'date_add' || $value == 'date_upd')
					unset($available_filters[$key]);

		//construct SQL filter
		$sql_filter = '';
		$sql_join = '';
		if ($this->urlFragments)
		{
			$schema = 'schema';
			// if we have to display the schema
			if (isset($this->urlFragments[$schema]))
			{
				if ($this->urlFragments[$schema] == 'blank' || $this->urlFragments[$schema] == 'synopsis')
				{
					$this->schemaToDisplay = $this->urlFragments[$schema];
					return true;
				}
				else
				{
					$this->setError(400, 'Please select a schema of type \'synopsis\' to get the whole schema informations (which fields are required, which kind of content...) or \'blank\' to get an empty schema to fill before using POST request', 28);
					return false;
				}
			}
			else
			{
				// if there are filters
				if (isset($this->urlFragments['filter']))
					foreach ($this->urlFragments['filter'] as $field => $url_param)
					{
						if ($field != 'sort' && $field != 'limit')
						{
							if (!in_array($field, $available_filters))
							{
								// if there are linked tables
								if (isset($this->resourceConfiguration['linked_tables']) && isset($this->resourceConfiguration['linked_tables'][$field]))
								{
									// contruct SQL join for linked tables
									$sql_join .= 'LEFT JOIN `'._DB_PREFIX_.pSQL($this->resourceConfiguration['linked_tables'][$field]['table']).'` '.pSQL($field).' ON (main.`'.pSQL($this->resourceConfiguration['fields']['id']['sqlId']).'` = '.pSQL($field).'.`'.pSQL($this->resourceConfiguration['fields']['id']['sqlId']).'`)'."\n";
					
									// construct SQL filter for linked tables
									foreach ($url_param as $field2 => $value)
									{
										if (isset($this->resourceConfiguration['linked_tables'][$field]['fields'][$field2]))
										{
											$linked_field = $this->resourceConfiguration['linked_tables'][$field]['fields'][$field2];
											$sql_filter .= $this->getSQLRetrieveFilter($linked_field['sqlId'], $value, $field.'.');
										}
										else
										{
											$list = array_keys($this->resourceConfiguration['linked_tables'][$field]['fields']);
											$this->setErrorDidYouMean(400, 'This filter does not exist for this linked table', $field2, $list, 29);
											return false;
										}
									}
								}
								elseif ($url_param != '' && in_array($field, $i18n_available_filters))
								{
									if (!is_array($url_param))
										$url_param = array($url_param);
									$sql_join .= 'LEFT JOIN `'._DB_PREFIX_.pSQL($this->resourceConfiguration['retrieveData']['table']).'_lang` main_i18n ON (main.`'.pSQL($this->resourceConfiguration['fields']['id']['sqlId']).'` = main_i18n.`'.pSQL($this->resourceConfiguration['fields']['id']['sqlId']).'`)'."\n";
									foreach ($url_param as $field2 => $value)
									{
										$linked_field = $this->resourceConfiguration['fields'][$field];
										$sql_filter .= $this->getSQLRetrieveFilter($linked_field['sqlId'], $value, 'main_i18n.');
										$language_filter = '['.implode('|',$this->_available_languages).']';
										$sql_filter .= $this->getSQLRetrieveFilter('id_lang', $language_filter, 'main_i18n.');
									}
								}
								// if there are filters on linked tables but there are no linked table
								elseif (is_array($url_param))
								{
									if (isset($this->resourceConfiguration['linked_tables']))
										$this->setErrorDidYouMean(400, 'This linked table does not exist', $field, array_keys($this->resourceConfiguration['linked_tables']), 30);
									else
										$this->setError(400, 'There is no existing linked table for this resource', 31);
									return false;
								}
								else
								{
									$this->setErrorDidYouMean(400, 'This filter does not exist', $field, $available_filters, 32);
									return false;
								}
							}
							elseif ($url_param == '')
							{
								$this->setError(400, 'The filter "'.$field.'" is malformed.', 33);
								return false;
							}
							else
							{
								if (isset($this->resourceConfiguration['fields'][$field]['getter']))
								{
									$this->setError(400, 'The field "'.$field.'" is dynamic. It is not possible to filter GET query with this field.', 34);
									return false;
								}
								else
								{
									if (isset($this->resourceConfiguration['retrieveData']['tableAlias'])) {
										$sql_filter .= $this->getSQLRetrieveFilter($this->resourceConfiguration['fields'][$field]['sqlId'], $url_param, $this->resourceConfiguration['retrieveData']['tableAlias'].'.');
									} else {
										$sql_filter .= $this->getSQLRetrieveFilter($this->resourceConfiguration['fields'][$field]['sqlId'], $url_param);
									}
								}
							}
						}
					}
			}
		}
		
		if (!$this->setFieldsToDisplay())
			return false;
		// construct SQL Sort
		$sql_sort = '';
		if (isset($this->urlFragments['sort']))
		{
			preg_match('#^\[(.*)\]$#Ui', $this->urlFragments['sort'], $matches);
			if (count($matches) > 1)
				$sorts = explode(',', $matches[1]);
			else
				$sorts = array($this->urlFragments['sort']);
	
			$sql_sort .= ' ORDER BY ';
			foreach ($sorts as $sort)
			{
				$delimiterPosition = strrpos($sort, '_');
				if ($delimiterPosition !== false)
				{
					$fieldName = substr($sort, 0, $delimiterPosition);
					$direction = strtoupper(substr($sort, $delimiterPosition + 1));
				}
				if ($delimiterPosition === false || !in_array($direction, array('ASC', 'DESC')))
				{
					$this->setError(400, 'The "sort" value has to be formed as this example: "field_ASC" or \'[field_1_DESC,field_2_ASC,field_3_ASC,...]\' ("field" has to be an available field)', 37);
					return false;
				}
				elseif (!in_array($fieldName, $available_filters) && !in_array($fieldName, $i18n_available_filters))
				{
					$this->setError(400, 'Unable to filter by this field. However, these are available: '.implode(', ', $available_filters).', for i18n fields:'.implode(', ', $i18n_available_filters), 38);
					return false;
				}
				// for sort on i18n field
				elseif (in_array($fieldName, $i18n_available_filters))
				{
					if (!preg_match('#main_i18n#', $sql_join))
						$sql_join .= 'LEFT JOIN `'._DB_PREFIX_.pSQL($this->resourceConfiguration['retrieveData']['table']).'_lang` main_i18n ON (main.`'.pSQL($this->resourceConfiguration['fields']['id']['sqlId']).'` = main_i18n.`'.pSQL($this->resourceConfiguration['fields']['id']['sqlId']).'`)'."\n";
					$sql_sort .= 'main_i18n.`'.pSQL($this->resourceConfiguration['fields'][$fieldName]['sqlId']).'` '.$direction.', ';// ORDER BY main_i18n.`field` ASC|DESC
				}
				else
				{
					$sql_sort .= (isset($this->resourceConfiguration['retrieveData']['tableAlias']) ? $this->resourceConfiguration['retrieveData']['tableAlias'].'.' : '').'`'.pSQL($this->resourceConfiguration['fields'][$fieldName]['sqlId']).'` '.$direction.', ';// ORDER BY `field` ASC|DESC
				}
			}
			$sql_sort = rtrim($sql_sort, ', ')."\n";
		}

		//construct SQL Limit
		$sql_limit = '';
		if (isset($this->urlFragments['limit']))
		{
			$limitArgs = explode(',', $this->urlFragments['limit']);
			if (count($limitArgs) > 2)
			{
				$this->setError(400, 'The "limit" value has to be formed as this example: "5,25" or "10"', 39);
				return false;
			}
			else
				$sql_limit .= ' LIMIT '.(int)($limitArgs[0]).(isset($limitArgs[1]) ? ', '.(int)($limitArgs[1]) : '')."\n";// LIMIT X|X, Y
		}
		$filters['sql_join'] = $sql_join;
		$filters['sql_filter'] = $sql_filter;
		$filters['sql_sort'] = $sql_sort;
		$filters['sql_limit'] = $sql_limit;
		
		return $filters;
	}
	
	public function getFilteredObjectList()
	{
		$objects = array();
		$filters = $this->manageFilters();

		/* If we only need to display the synopsis, analyzing the first row is sufficient */
		if (isset($this->urlFragments['schema']) && in_array($this->urlFragments['schema'], array('blank', 'synopsis')))
			$filters = array('sql_join' => '', 'sql_filter' => '', 'sql_sort' => '', 'sql_limit' => ' LIMIT 1');

		$this->resourceConfiguration['retrieveData']['params'][] = $filters['sql_join'];
		$this->resourceConfiguration['retrieveData']['params'][] = $filters['sql_filter'];
		$this->resourceConfiguration['retrieveData']['params'][] = $filters['sql_sort'];
		$this->resourceConfiguration['retrieveData']['params'][] = $filters['sql_limit'];
		
		$tmp = new $this->resourceConfiguration['retrieveData']['className']();
		$sqlObjects = call_user_func_array(array($tmp, $this->resourceConfiguration['retrieveData']['retrieveMethod']), $this->resourceConfiguration['retrieveData']['params']);
		if ($sqlObjects)
		{
			foreach ($sqlObjects as $sqlObject)
				$objects[] = new $this->resourceConfiguration['retrieveData']['className']($sqlObject[$this->resourceConfiguration['fields']['id']['sqlId']]);
			return $objects;
		}
	}
	
	public function getFilteredObjectDetails()
	{
		$objects = array();
		if (!isset($this->urlFragments['display'])) 
			$this->fieldsToDisplay = 'full';
		//get entity details
		$object = new $this->resourceConfiguration['retrieveData']['className']($this->urlSegment[1]);
		if ($object->id)
		{
			$objects[] = $object;
			return $objects;
		}
		elseif (!count($this->errors))
		{
			$this->objOutput->setStatus(404);
			$this->_outputEnabled = false;
			return false;
		}
	}
	
	/**
	 * Execute GET and HEAD requests
	 * 
	 * Build filter
	 * Build fields display
	 * Build sort
	 * Build limit
	 * 
	 * @return boolean
	 */
	public function executeEntityGetAndHead()
	{
		if ($this->resourceConfiguration['objectsNodeName'] != 'resources')
		{
			if (!isset($this->urlSegment[1]) || !strlen($this->urlSegment[1]))
				$return = $this->getFilteredObjectList();
			else
				$return = $this->getFilteredObjectDetails();
			
			if (!$return)
				return false;
			else
				$this->objects = $return;
		}
		return true;
	}
	
	/**
	 * Execute POST method on a PrestaShop entity
	 * 
	 * @return boolean
	 */
	protected function executeEntityPost()
	{
		if ($this->outputFormat == 'JSON')
			return $this->saveEntityFromJSON(201);
		else
			return $this->saveEntityFromXml(201);
	}
	
	public function saveEntityFromJSON($code)
	{
		// not implemented yet
	}

	/**
	 * Execute PUT method on a PrestaShop entity
	 * 
	 * @return boolean
	 */
	protected function executeEntityPut()
	{
		return $this->saveEntityFromXml(200);

	}
	
	/**
	 * Execute DELETE method on a PrestaShop entity
	 * 
	 * @return boolean
	 */
	protected function executeEntityDelete()
	{
		$objects = array();
		$arr_avoid_id = array();
		$ids = array();
		if (isset($this->urlFragments['id']))
		{
			preg_match('#^\[(.*)\]$#Ui', $this->urlFragments['id'], $matches);
			if (count($matches) > 1)
				$ids = explode(',', $matches[1]);
		}
		else
		{
			$ids[] = (int)$this->urlSegment[1];
		}
		if (!empty($ids))
		{
			foreach ($ids as $id)
			{
				$object = new $this->resourceConfiguration['retrieveData']['className']((int)$id);
				if (!$object->id)
					$arr_avoid_id[] = $id;
				else
					$objects[] = $object;
			}
		}
		
		if (!empty($arr_avoid_id) || empty($ids))
		{
			$this->setError(404, 'Id(s) not exists: '.implode(', ', $arr_avoid_id), 87);
			$this->_outputEnabled = true;
		}
		else 
		{
			foreach ($objects as $object)
			{
				if (isset($this->resourceConfiguration['objectMethods']) && isset($this->resourceConfiguration['objectMethods']['delete']))
					$result = $object->{$this->resourceConfiguration['objectMethods']['delete']}();
				else
					$result = $object->delete();
				
				if (!$result)
					$arr_avoid_id[] = $object->id;
			}
			if (!empty($arr_avoid_id))
			{
				$this->setError(500, 'Id(s) wasn\'t deleted: '.implode(', ', $arr_avoid_id), 88);
				$this->_outputEnabled = true;
			}
			else
				$this->_outputEnabled = false;
		}
	}
	
	/**
	 * Write XML output after GET and HEAD action
	 * 
	 * @return void
	 */
		
	/**
	 * save Entity Object from XML
	 * 
	 * @param int $successReturnCode
	 * @return boolean
	 */
	protected function saveEntityFromXml($successReturnCode)
	{
		try
		{
			$xml = new SimpleXMLElement($this->_inputXml);
		}
		catch (Exception $error)
		{
			$this->setError(500, 'XML error : '.$error->getMessage()."\n".'XML length : '.strlen($this->_inputXml)."\n".'Original XML : '.$this->_inputXml, 127);
			return;
		}
		$xmlEntities = $xml->children();
		$object = null;
		
		$ids = array();
		foreach ($xmlEntities as $entity) 
		{
			// To cast in string allow to check null values
			if ((string)$entity->id != '')
				$ids[] = (int)$entity->id;
		}
		if ($this->method == 'PUT')
		{
			$ids2 = array();
			$ids2 = array_unique($ids);
			if (count($ids2) != count($ids))
			{
				$this->setError(400, 'id is duplicate in request', 89);
				return false;
			}
			if (count($xmlEntities) != count($ids))
			{
				$this->setError(400, 'id is required when modifying a resource', 90);
				return false;
			}
		}
		elseif ($this->method == 'POST' && count($ids) > 0)
		{
			$this->setError(400, 'id is forbidden when adding a new resource', 91);
			return false;
		}
		
		foreach ($xmlEntities as $xmlEntity)
		{
			$attributes = $xmlEntity->children();
			
			if ($this->method == 'POST')
				$object = new $this->resourceConfiguration['retrieveData']['className']();
			elseif ($this->method == 'PUT')
			{
				$object = new $this->resourceConfiguration['retrieveData']['className']((int)$attributes->id);
				if (!$object->id)
				{
					$this->setError(404, 'Invalid ID', 92);
					return false;
				}
			}
			$this->objects[] = $object;
			$i18n = false;
			// attributes
			foreach ($this->resourceConfiguration['fields'] as $fieldName => $fieldProperties)
			{
				$sqlId = $fieldProperties['sqlId'];
				
				if ($fieldName == 'id')
					$sqlId = $fieldName;
				if (isset($attributes->$fieldName) && isset($fieldProperties['sqlId']) && (!isset($fieldProperties['i18n']) || !$fieldProperties['i18n']))
				{
					if (isset($fieldProperties['setter']))
					{
						// if we have to use a specific setter
						if (!$fieldProperties['setter'])
						{
							// if it's forbidden to set this field
							$this->setError(400, 'parameter "'.$fieldName.'" not writable. Please remove this attribute of this XML', 93);
							return false;
						}
						else
							$object->$fieldProperties['setter']((string)$attributes->$fieldName);
					}
					elseif (Tools::property_exists($object, $sqlId))
						$object->$sqlId = (string)$attributes->$fieldName;
					else
						$this->setError(400, 'Parameter "'.$fieldName.'" cannot be set to the object "'.$this->resourceConfiguration['retrieveData']['className'].'"', 123);
						
				}
				elseif (isset($fieldProperties['required']) && $fieldProperties['required'] && !$fieldProperties['i18n'])
				{
					$this->setError(400, 'parameter "'.$fieldName.'" required', 41);
					return false;
				}
				elseif ((!isset($fieldProperties['required']) || !$fieldProperties['required']) && Tools::property_exists($object, $sqlId))
					$object->$sqlId = null;
				
				if (isset($fieldProperties['i18n']) && $fieldProperties['i18n'])
				{
					$i18n = true;
					if (isset($attributes->$fieldName, $attributes->$fieldName->language))
						foreach ($attributes->$fieldName->language as $lang)
							$object->{$fieldName}[(int)$lang->attributes()->id] = (string)$lang;
				}
			}
					
			if (!$this->hasErrors())
			{
				if ($i18n && ($retValidateFieldsLang = $object->validateFieldsLang(false, true)) !== true)
				{
					$this->setError(400, 'Validation error: "'.$retValidateFieldsLang.'"', 84);
					return false;
				}
				elseif (($retValidateFields = $object->validateFields(false, true)) !== true)
				{
					$this->setError(400, 'Validation error: "'.$retValidateFields.'"', 85);
					return false;
				}
				else
				{
					// Call alternative method for add/update
					$objectMethod = ($this->method == 'POST' ? 'add' : 'update');
					if (isset($this->resourceConfiguration['objectMethods']) && array_key_exists($objectMethod, $this->resourceConfiguration['objectMethods']))
						$objectMethod = $this->resourceConfiguration['objectMethods'][$objectMethod];
					$result = $object->{$objectMethod}();
					if ($result)
					{
						if (isset($attributes->associations))
							foreach ($attributes->associations->children() as $association)
							{
								// associations
								if (isset($this->resourceConfiguration['associations'][$association->getName()]))
								{
									$assocItems = $association->children();
									$values = array();
									foreach ($assocItems as $assocItem)
									{
										$fields = $assocItem->children();
										$entry = array();
										foreach ($fields as $fieldName => $fieldValue)
										{
											if (isset($fieldValue->language))
											{
												foreach ($fieldValue->language as $lang)
													$entry[$fieldName][(int)$lang->attributes()->id] = (string)$lang;
											}
											else
												$entry[$fieldName] = (string)$fieldValue;
										}
										$values[] = $entry;
									}
									$setter = $this->resourceConfiguration['associations'][$association->getName()]['setter'];
									if (!is_null($setter) && $setter && method_exists($object, $setter) && !$object->$setter($values))
									{
										$this->setError(500, 'Error occurred while setting the '.$association->getName().' value', 85);
										return false;
									}
								}
								elseif ($association->getName() != 'i18n')
								{
									$this->setError(400, 'The association "'.$association->getName().'" does not exists', 86);
									return false;
								}
							}

					}
					else
						$this->setError(500, 'Unable to save resource', 46);
				}
			}
		}
		if (!$this->hasErrors())
		{
			$this->objOutput->setStatus($successReturnCode);
			return true;
		}
	}
	
	/**
	 * get SQL retrieve Filter
	 * 
	 * @param string $sqlId
	 * @param string $filterValue
	 * @param string $tableAlias = 'main.'
	 * @return string
	 */
	protected function getSQLRetrieveFilter($sqlId, $filterValue, $tableAlias = 'main.')
	{
		$ret = '';
		preg_match('/^(.*)\[(.*)\](.*)$/', $filterValue, $matches);
		if (count($matches) > 1)
		{
			if ($matches[1] == '%' || $matches[3] == '%')
				$ret .= ' AND '.$tableAlias.'`'.pSQL($sqlId).'` LIKE "'.$matches[1].pSQL($matches[2]).$matches[3]."\"\n";
			elseif ($matches[1] == '' && $matches[3] == '')
			{
				if (strpos($matches[2], '|') > 0)
				{
					$values = explode('|', $matches[2]);
					$ret .= ' AND (';
					$temp = '';
					foreach ($values as $value)
						$temp .= $tableAlias.'`'.pSQL($sqlId).'` = "'.pSQL($value).'" OR ';
					$ret .= rtrim($temp, 'OR ').')'."\n";
				}
				elseif (preg_match('/^([\d\.:-\s]+),([\d\.:-\s]+)$/', $matches[2], $matches3))
				{
					unset($matches3[0]);
					if (count($matches3) > 0)
					{
						sort($matches3);
						$ret .= ' AND '.$tableAlias.'`'.pSQL($sqlId).'` BETWEEN "'.pSQL($matches3[0]).'" AND "'.pSQL($matches3[1])."\"\n";
					}
				}
				else
					$ret .= ' AND '.$tableAlias.'`'.pSQL($sqlId).'`="'.pSQL($matches[2]).'"'."\n";
			}
			elseif ($matches[1] == '>')
				$ret .= ' AND '.$tableAlias.'`'.pSQL($sqlId).'` > "'.pSQL($matches[2])."\"\n";
			elseif ($matches[1] == '<')
				$ret .= ' AND '.$tableAlias.'`'.pSQL($sqlId).'` < "'.pSQL($matches[2])."\"\n";
			elseif ($matches[1] == '!')
				$ret .= ' AND '.$tableAlias.'`'.pSQL($sqlId).'` != "'.pSQL($matches[2])."\"\n";
		}
		else
			$ret .= ' AND '.$tableAlias.'`'.pSQL($sqlId).'` '.(Validate::isFloat(pSQL($filterValue)) ? 'LIKE' : '=').' "'.pSQL($filterValue)."\"\n";
		return $ret;
	}
	
	public function filterLanguage()
	{
		$arr_languages = array();
		$length_values = strlen($this->urlFragments['language']);
		// if just one language is asked
		if (is_numeric($this->urlFragments['language']))
			$arr_languages[] = (int)$this->urlFragments['language'];
		// if a range or a list is asked
		elseif (strpos($this->urlFragments['language'], '[') === 0
			&& strpos($this->urlFragments['language'], ']') === $length_values-1)
		{
			if (strpos($this->urlFragments['language'], '|') !== false
				XOR strpos($this->urlFragments['language'], ',') !== false)
			{
				$params_values = str_replace(array(']', '['), '', $this->urlFragments['language']);
				// it's a list
				if (strpos($params_values, '|') !== false)
				{
					$list_enabled_lang = explode('|', $params_values);
					$arr_languages = $list_enabled_lang;
				}
				// it's a range
				elseif (strpos($params_values, ',') !== false)
				{
					$range_enabled_lang = explode(',', $params_values);
					if (count($range_enabled_lang) != 2)
					{
						$this->setError(400, 'A range value for a language must contains only 2 values', 78);
						return false;
					}
					for ($i = $range_enabled_lang[0]; $i <= $range_enabled_lang[1]; $i++)
						$arr_languages[] = $i;
				}
			}
			else if (preg_match('#\[(\d)+\]#Ui', $this->urlFragments['language'], $match_lang))
			{
				$arr_languages[] = $match_lang[1];
			}
		}
		else
		{
			$this->setError(400, 'language value is wrong', 79);
			return false;
		}

		$result = array_map('is_numeric', $arr_languages);
		if (array_search(false, $result, true))
		{
			$this->setError(400, 'Language ID must be numeric', 80);
			return false;
		}
		
		foreach ($arr_languages as $key=>$id_lang)
			if (!Language::getLanguage($id_lang))
				unset($arr_languages[$key]);
		
		return $arr_languages;
	}
	
	/**
	 * Thanks to the (WebserviceOutputBuilder) WebserviceKey::objOutput 
	 * Method build the output depend on the WebserviceRequest::outputFormat
	 * and set HTTP header parameters.
	 * 
	 * @return array with displaying informations (used in the dispatcher).
	 */
	protected function returnOutput()
	{
		$return = array();

		// write headers
		$this->objOutput->setHeaderParams('Access-Time', time())
						->setHeaderParams('X-Powered-By', 'PrestaShop Webservice')
						->setHeaderParams('PSWS-Version', _PS_VERSION_)
						->setHeaderParams('Execution-Time', round(microtime(true) - $this->_startTime,3));
		
		
		$return['type'] = strtolower($this->outputFormat);
		
		// write this header only now (avoid hackers happiness...)
		if ($this->_authenticated)
		{
			$this->objOutput->setHeaderParams('PSWS-Version', _PS_VERSION_);
		}
		
		// If Specific Management is asked
		if ($this->objectSpecificManagement instanceof WebserviceSpecificManagementInterface)
		{
			try {
				$return['content'] = $this->objectSpecificManagement->getContent();
			} catch (WebserviceException $e) {
				if ($e->getType() == WebserviceException::DID_YOU_MEAN)
					$this->setErrorDidYouMean($e->getStatus(), $e->getMessage(), $e->getWrongValue(), $e->getAvailableValues(), $e->getCode());
				elseif ($e->getType() == WebserviceException::SIMPLE)
					$this->setError($e->getStatus(), $e->getMessage(), $e->getCode());
			}
		}
		
		// for use a general output
		if (!$this->hasErrors() && $this->objectSpecificManagement == null)
		{
			if (empty($this->objects))
			{
				try {
					$return['content'] = $this->objOutput->getResourcesList($this->keyPermissions);
				} catch (WebserviceException $e) {
					if ($e->getType() == WebserviceException::DID_YOU_MEAN)
						$this->setErrorDidYouMean($e->getStatus(), $e->getMessage(), $e->getWrongValue(), $e->getAvailableValues(), $e->getCode());
					elseif ($e->getType() == WebserviceException::SIMPLE)
						$this->setError($e->getStatus(), $e->getMessage(), $e->getCode());
				}
			}
			else
			{
				try {
					if (isset($this->urlSegment[1]) && !empty($this->urlSegment[1]))
						$type_of_view = WebserviceOutputBuilder::VIEW_DETAILS;
					else
						$type_of_view = WebserviceOutputBuilder::VIEW_LIST;
					
					if (in_array($this->method, array('PUT', 'POST')))
					{
						$type_of_view = WebserviceOutputBuilder::VIEW_DETAILS;
						$this->fieldsToDisplay = 'full';
					}	
					
					$return['content'] = $this->objOutput->getContent($this->objects, $this->schemaToDisplay, $this->fieldsToDisplay, $this->depth, $type_of_view);
				} catch (WebserviceException $e) {
					if ($e->getType() == WebserviceException::DID_YOU_MEAN)
						$this->setErrorDidYouMean($e->getStatus(), $e->getMessage(), $e->getWrongValue(), $e->getAvailableValues(), $e->getCode());
					elseif ($e->getType() == WebserviceException::SIMPLE)
						$this->setError($e->getStatus(), $e->getMessage(), $e->getCode());
				} catch (Exception $e) {
					$this->setError(500, $e->getMessage(), $e->getCode());
				}
			}
		}
		
		
		// if the output is not enable, delete the content
		// the type content too
		if (!$this->_outputEnabled)
		{
			if (isset($return['type']))
				unset($return['type']);
			if (isset($return['content']))
				unset($return['content']);
		}
			
		elseif (isset($return['content']))
			$this->objOutput->setHeaderParams('Content-Sha1', sha1($return['content']));
		
		// if errors happends when creating returned xml, 
		// the usual xml content is replaced by the nice error handler content
		if ($this->hasErrors())
		{
			$this->_outputEnabled = true;
			$return['content'] = $this->objOutput->getErrors($this->errors);
		}
		
		if (!isset($return['content']) || strlen($return['content']) <= 0)
			$this->objOutput->setHeaderParams('Content-Type', '');

		$return['headers'] = $this->objOutput->buildHeader();
		restore_error_handler();

		return $return;
	}
}